<?php

namespace App\Http\Controllers;

use App\Models\TermsAndCondition;
use Illuminate\Http\Request;
use Exception;

class TermsAndConditionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $terms = TermsAndCondition::latest()->paginate(15);
        return view('terms.index', compact('terms'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('terms.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'identifier' => ['required', 'string', 'in:terms,privacy'],
                'title' => ['required', 'string'],
                'body' => ['required', 'string'],
                'title_ar' => ['nullable', 'string'],
                'body_ar' => ['nullable', 'string'],
                'is_active' => ['nullable', 'boolean'],
            ]);

            // Deactivate all existing terms/privacy with the same identifier
            if ($request->is_active) {
                TermsAndCondition::where('identifier', $request->identifier)->update(['is_active' => false]);
            }

            TermsAndCondition::create([
                'identifier' => $request->identifier,
                'title' => $request->title,
                'body' => $request->body,
                'title_ar' => $request->title_ar,
                'body_ar' => $request->body_ar,
                'is_active' => $request->is_active ?? true,
            ]);

            return redirect()->route('terms.index')->with('success', 'Terms and Conditions created successfully');
        } catch (Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\TermsAndCondition  $term
     * @return \Illuminate\Http\Response
     */
    public function show(TermsAndCondition $term)
    {
        return view('terms.show', compact('term'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\TermsAndCondition  $term
     * @return \Illuminate\Http\Response
     */
    public function edit(TermsAndCondition $term)
    {
        return view('terms.edit', compact('term'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\TermsAndCondition  $term
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, TermsAndCondition $term)
    {
        try {
            $request->validate([
                'identifier' => ['required', 'string', 'in:terms,privacy'],
                'title' => ['required', 'string'],
                'body' => ['required', 'string'],
                'title_ar' => ['nullable', 'string'],
                'body_ar' => ['nullable', 'string'],
                'is_active' => ['nullable', 'boolean'],
            ]);

            // If activating this terms/privacy, deactivate all others with the same identifier
            if ($request->is_active) {
                TermsAndCondition::where('identifier', $request->identifier)
                    ->where('id', '!=', $term->id)
                    ->update(['is_active' => false]);
            }

            $term->update($request->all());

            return redirect()->route('terms.index')->with('success', 'Terms and Conditions updated successfully');
        } catch (Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\TermsAndCondition  $term
     * @return \Illuminate\Http\Response
     */
    public function destroy(TermsAndCondition $term)
    {
        try {
            $term->delete();
            return redirect()->route('terms.index')->with('success', 'Terms and Conditions deleted successfully');
        } catch (Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }
}

