<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Hospital;
use App\Models\HospitalSpeciality;
use App\Models\Speciality;
use App\Models\Doctor;
use Exception;
use Illuminate\Support\Facades\DB;

class HospitalController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $hospitals = Hospital::get();
        return view('hospitals.index', compact('hospitals'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $specialties = Speciality::get();
        return view('hospitals.create', compact('specialties'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $hospital =  Hospital::create([
                'name' => $request->name,
                'description' => $request->description,
                'phone' => $request->phone,
                'address' => $request->address,
                'from' => $request->from,
                'to' => $request->to,
                'created_by' => auth()->user()->id,
            ]);

            if ($request->hasFile('image')) {
                $hospital->addMediaFromRequest('image')
                    ->toMediaCollection($hospital->id);
            }

            foreach ($request->specialties as $specialtyId) {
                HospitalSpeciality::create([
                    'hospital_id' => $hospital->id,
                    'specialty_id' => $specialtyId,
                ]);
            }


            return redirect()->route('hospitals.index')->with('success', 'Hospital Created Successfully');
        } catch (Exception $ex) {
            return redirect()->back()->with('error', $ex->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Hospital $hospital)
    {
        try {
            return view('hospitals.show', compact('hospital'));
        } catch (Exception $ex) {
            return redirect()->back()->with('error', $ex->getMessage());
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Request $request, Hospital $hospital)
    {
        $hospital = Hospital::findOrFail($hospital->id);
        $specialties = Speciality::all();
        $doctors = Doctor::all();
        $assignedDoctors = DB::table('doctor_hospital')
        ->where('hospital_id', $hospital->id)
        ->get();
        
        return view('hospitals.edit', compact('hospital', 'specialties', 'doctors', 'assignedDoctors'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Hospital $hospital)
    {
        try {
            $hospital->update([
                'name' => $request->name,
                'address' => $request->address,
                'phone' => $request->phone,
                'from' => $request->from,
                'to' => $request->to,
                'description' => $request->description,
            ]);
            
                if ($request->hasFile('image')) {
                    $hospital->clearMediaCollection($hospital->id);
            
                    $hospital->addMediaFromRequest('image')
                             ->toMediaCollection($hospital->id);
                }


                // Reassign if there is assignedDoctors in request
                if ($request->has('assignedDoctors')) {
                DB::table('doctor_hospital')->where('hospital_id', $hospital->id)->delete();
                    foreach ($request->assignedDoctors as $doctor) {
                        DB::table('doctor_hospital')->insert([
                            'doctor_id' => $doctor['doctor_id'],
                            'hospital_id' => $hospital->id,
                            'am' => json_encode(['from' => $doctor['am_from'], 'to' => $doctor['am_to']]),
                            'pm' => json_encode(['from' => $doctor['pm_from'], 'to' => $doctor['pm_to']]),
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }
                }
            
            return redirect()->route('hospitals.index')->with('success', 'Hospital Updated Successfully');
        } catch (Exception $ex) {
            return redirect()->back()->with('error', $ex->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Hospital $hospital)
    {
        try {
            $hospital->delete();
            return redirect()->route('hospitals.index')->with('success', 'Hospital Deleted Successfully');
        } catch (Exception $ex) {
            return redirect()->back()->with('error', $ex->getMessage());
        }
    }
    
    public function assign()
    {
        $doctors = Doctor::all();
        $hospitals = Hospital::all();

        return view('hospitals.assign', compact('doctors', 'hospitals'));
    }
    
    public function assignStore(Request $request)
    {
        $doctorId = $request->input('doctor');
        $hospitalId = $request->input('hospital');
    
        $am = json_encode([
            'from' => $request->amfrom,
            'to' => $request->amto
        ]);
    
        $pm = json_encode([
            'from' => $request->pmfrom,
            'to' => $request->pmto
        ]);
    
        DB::table('doctor_hospital')->insert([
            'doctor_id' => $doctorId,
            'hospital_id' => $hospitalId,
            'am' => $am,
            'pm' => $pm,
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    
        return redirect()->back()->with('success', 'Doctor assigned to Hospital successfully.');
    }

}
