<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Doctor;
use App\Models\File;
use App\Models\Certification;
use Exception;
use App\Http\Resources\DoctorResource;
use App\Http\Resources\FileResource;
use App;
use App\Models\Speciality;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class DoctorController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        //
        try {
            $lang = $request->header('lang');
            $doctors = Doctor::where('status', 'accepted')->get();
            $data = DoctorResource::collection($doctors);

            return response()->json([
                'statusCode' => 200,
                'data' => $data
            ]);
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Request $request)
    {
        try {
            $lang = $request->header('lang');

            $doctor = Doctor::find($request->id);
            if (!$doctor) {
                return response()->json([
                    'statusCode' => 404,
                    'message' => $lang == 'en' ? 'Invalid Doctor Id ' : 'رقم الدكتور غير معرف',
                ]);
            }

            $getFiles = $doctor->getMedia($doctor->phone)->filter(function ($media) {
                // Filter for files (e.g., PDF, Word documents, etc.)
                return strstr($media->mime_type, 'application/') !== false;
            });

            $image = $doctor->getMedia($doctor->phone)->first(function ($media) {
                // Filter for images (e.g., JPEG, PNG, GIF, etc.)
                return strstr($media->mime_type, 'image/') !== false;
            });


            $data = array();
            $clinics = array();
            $hospitals = [];

            $files = array();
            foreach ($getFiles as $file) {
                $files[] = $file->getUrl();
            }

            foreach ($doctor->clinics as $clinic) {
                array_push($clinics, [
                    'clinic_name' => $clinic->name,
                    'clinic_address' => $clinic->address,
                    'clinic_phone' => $clinic->phone,
                    'clinic_rate' => $clinic->rate,
                    'clinic_description' => $clinic->description,
                    'am' => [
                        'from' => json_decode($clinic->pivot->am)->from,
                        'to' => json_decode($clinic->pivot->am)->to
                    ],
                    'pm' => [
                        'from' => json_decode($clinic->pivot->pm)->from,
                        'to' => json_decode($clinic->pivot->pm)->to
                    ],
                ]);
            };
            
            foreach ($doctor->hospitals as $hospital) {
                $hospitals[] = [
                    'hospital_name' => $hospital->name,
                    'hospital_address' => $hospital->address,
                    'hospital_phone' => $hospital->phone,
                    'hospital_description' => $hospital->description,
                    'am' => [
                        'from' => json_decode($hospital->pivot->am)->from,
                        'to' => json_decode($hospital->pivot->am)->to,
                    ],
                    'pm' => [
                        'from' => json_decode($hospital->pivot->pm)->from,
                        'to' => json_decode($hospital->pivot->pm)->to,
                    ],
                ];
            }

            array_push($data, [
                'id' => $doctor->id,
                'name' => $doctor->name,
                'image' => $image ? $image->getUrl() : asset('images/anime3.png'),
                'specialty' => $lang == 'ar' ? $doctor->specialty->name_ar ?? '' : $doctor->specialty->name_en ?? '',
                'phone' => $doctor->phone,
                'email' => $doctor->email,
                'gender' => $doctor->gender,
                'job_title' => $doctor->job_title,
                'summary' => $doctor->summary,
                'state' => $doctor->state,
                'city' => $doctor->city,
                'ragion' => $doctor->ragion,
                'secretary_number' => $doctor->secretary_number,
                'files' => $files,
                'clinics' => $clinics,
                'hospitals' => $hospitals,
            ]);


            return response()->json([
                'statusCode' => 200,
                'data' => $data[0]
            ]);
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request)
    {
        try {
            $lang = $request->header('lang');

            if ($request->type == 'doctor') {

                $doctor_id = $request->id;

                if (!$doctor_id || !Doctor::find($doctor_id)) {
                    return response()->json([
                        'statusCode' => 404,
                        'message' => $lang == 'en' ? 'Invalid Doctor Id ' : 'رقم الدكتور غير معرف',
                    ]);
                }
                $doctor = Doctor::find($request->id);

                // Remove existing media (profile picture) if any
                $doctor->clearMediaCollection($doctor->phone);

                // Upload and associate the new profile picture
                $doctor->addMediaFromRequest('image')
                    ->toMediaCollection($doctor->phone);

                DB::beginTransaction();
                $doctor->update($request->except(['type']));
                DB::commit();

                return response()->json([
                    'statusCode' => 200,
                    'message' => $lang == 'en' ? 'Doctor Updated Successfully' : 'تم تعديل بيانات الدكتور بنجاح',
                ]);
            }
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request)
    {
        try {
            $lang = $request->header('lang');
            $doctor = Doctor::find($request->id);

            if (!$doctor) {
                return response()->json([
                    'statusCode' => 404,
                    'message' => $lang == 'en' ? 'Invalid Doctor Id ' : 'رقم الدكتور غير معرف',
                ]);
            }
            $doctor->delete();

            return response()->json([
                'statusCode' => 200,
                'message' => $lang == 'en' ? 'Doctor has been Deleted Successfully' : 'تم حذف بيانات الدكتور بنجاح',
            ]);
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }

    public function doctorImage(Request $request)
    {
        $lang = $request->header('lang');

        try {
            $this->validate($request, [
                'image' => 'required|image|mimes:jpg,png,jpeg,gif,svg',
            ]);
            $doctor = Doctor::find($request->id);


            if (!$doctor) {
                return response()->json([
                    'statusCode' => 404,
                    'message' => $lang == 'en' ? 'Invalid Doctor Id ' : 'رقم الدكتور غير معرف',
                ]);
            }

            // Remove existing media (profile picture) if any
            $doctor->clearMediaCollection($doctor->phone);

            // Upload and associate the new profile picture
            $doctor->addMediaFromRequest('image')
                ->toMediaCollection($doctor->phone);

            $image = $doctor->getMedia($doctor->phone)->first(function ($media) {
                // Filter for images (e.g., JPEG, PNG, GIF, etc.)
                return strstr($media->mime_type, 'image/') !== false;
            });


            return response([
                'statusCode' => 200,
                'message' => $lang == 'ar' ? 'تم تغيير الصورة بنجاح' : 'image uploaded successfully',
                'data' => [
                    'image' => $image->getUrl(),
                ]
            ]);
        } catch (Exception $e) {
            return response([
                'statusCode' => 422,
                'message' => $lang == 'en' ? 'files is not allowed' : 'فشل فى تحميل الملفات',
            ]);
        }
    }

    public function fileUpload(Request $request)
    {
        $doctor = Doctor::find($request->doctor_id);
        $lang = $request->header('lang');

        if (!$doctor) {
            return response()->json([
                'statusCode' => 404,
                'message' => $lang == 'en' ? 'Invalid Doctor Id ' : 'رقم الدكتور غير معرف',
            ]);
        }
        try {
            if (isset($request['files'])) {

                for ($i = 0; $i < count($request['files']); $i++) {
                    $fileName = str_replace(" ", "", $request['files'][$i]->getClientOriginalName()) . '.' . $request['files'][$i]->extension();
                    $request['files'][$i]->move(public_path('images'), $fileName);

                    /* Store $fileName name in DATABASE from HERE */
                    $file = new File();
                    $file->doctor_id = $request->doctor_id;
                    $file->name = $fileName;
                    $file->save();
                }
            }
            return response()->json([
                'statusCode' => 200,
                'message' => $lang == 'en' ? 'files uploaded successfully' : 'تم رفع الملفات بنجاح',
            ]);
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }

    public function getFiles(Request $request)
    {
        try {
            $lang = $request->header('lang');

            $doctor = Doctor::find($request->id);
            $files = $doctor->files;
            if (!$doctor) {
                return response()->json([
                    'statusCode' => 404,
                    'message' => $lang == 'en' ? 'Invalid Doctor Id ' : 'رقم الدكتور غير معرف',
                ]);
            } else {
                if (!$files) {
                    return response()->json([
                        'statusCode' => 404,
                        'message' => $lang == 'en' ? 'Doctor has no files ' : 'غير متوفر ملفات',
                    ]);
                } else {
                    $data = FileResource::collection($files);
                    return response()->json([
                        'statusCode' => 200,
                        'data' => $data,
                    ]);
                }
            }
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }


    public function showDoctors($id)
    {
        try {
            $doctors = Doctor::where('specialty_id', $id)->get();
            $data = DoctorResource::collection($doctors);

            return response()->json([
                'data' => $data,
                'statusCode' => 200
            ]);
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }

    public function filterDoctors(Request $request)
    {
        try {
            $doctorquery = Doctor::query();
            $specialtyquery = Speciality::query();

            // Check if the request contains a specialty ID
            if ($request->name != null) {
            $specialtyquery->where('name_en', 'like', '%' . $request->input('name') . '%')->orWhere('name_ar', 'like', '%' . $request->input('name') . '%');
                $specialty = $specialtyquery->first();
                if ($specialty != null) {
                    $doctorquery->where('specialty_id', $specialty->id);
                    $doctors = $doctorquery->get();
                    $data =  DoctorResource::collection($doctors);
                    return response()->json([
                        'data' => $data,
                        'statusCode' => 200
                    ]);
                }
            }

            // Check if the request contains a search query for doctor name
            if ($request->name != null) {
            $doctorquery->where('name', 'like', '%' . $request->input('name') . '%')->orWhere('state', 'like', '%' . $request->input('name') . '%')->orWhere('city', 'like', '%' . $request->input('name') . '%')->orWhere('ragion', 'like', '%' . $request->input('name') . '%');
                $doctors = $doctorquery->get();
                $data =  DoctorResource::collection($doctors);
                return response()->json([
                    'data' => $data,
                    'statusCode' => 200
                ]);
            }

            $doctors = [];

            return response()->json([
                'data' => $doctors,
                'statusCode' => 200
            ]);
        } catch (Exception $e) {
            return response()->json([
                'statusCode' => 500,
                'message' => $e->getMessage(),
            ]);
        }
    }
}
